"use client"

import React, { useState } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Checkbox } from "@/components/ui/checkbox"
import {
  Menu,
  X,
  ArrowRight,
  Mail,
  MapPin,
  Clock,
  Send,
  CheckCircle,
  Linkedin,
  Instagram,
  Phone,
  Plus,
  Minus,
} from "lucide-react"
import Image from "next/image"
import Link from "next/link"

export default function ContactPage() {
  const [isMenuOpen, setIsMenuOpen] = useState(false)
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [showSuccess, setShowSuccess] = useState(false)
  const [openFaq, setOpenFaq] = useState<number | null>(null)

  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault()
    setIsSubmitting(true)

    // Simulate form submission
    await new Promise((resolve) => setTimeout(resolve, 2000))

    setIsSubmitting(false)
    setShowSuccess(true)

    // Reset form after showing success
    setTimeout(() => {
      setShowSuccess(false)
      ;(e.target as HTMLFormElement).reset()
    }, 3000)
  }

  const faqs = [
    {
      question: "How quickly do you respond to project inquiries?",
      answer:
        "I typically respond to all project inquiries within 24 hours during business days. For urgent projects or time-sensitive opportunities, you can reach me directly via phone or LinkedIn for faster communication.",
    },
    {
      question: "Do you work with international clients and remote teams?",
      answer:
        "I work with clients worldwide and have extensive experience collaborating across different time zones. I use modern communication tools and project management systems to ensure seamless collaboration regardless of location.",
    },
    {
      question: "What information should I include in my project inquiry?",
      answer:
        "Please include your project goals, target audience, timeline, budget range, and any specific requirements or features you have in mind. The more details you provide about your vision and business objectives, the better I can tailor my proposal to meet your needs.",
    },
    {
      question: "Do you offer ongoing support after project completion?",
      answer:
        "Yes, I provide comprehensive post-launch support including maintenance, updates, performance monitoring, and feature enhancements. I believe in building long-term partnerships and ensuring your digital solution continues to evolve with your business.",
    },
  ]

  return (
    <div className="min-h-screen bg-white text-gray-900">
      <header className="sticky top-0 z-50 backdrop-blur-3xl bg-[#161d25]/95 border-b border-white/10 shadow-2xl">
        <div className="container mx-auto px-4 py-6 relative">
          <div className="flex items-center justify-between">
            <Link href="/" className="flex items-center gap-4">
              <div className="relative group">
                <div className="absolute inset-0 bg-gradient-to-br from-gray-500/20 to-gray-400/10 rounded-3xl blur-xl group-hover:blur-2xl transition-all duration-700"></div>
                <Image
                  src="https://i.ibb.co/5XLC8VTC/kdsgnpng2.png"
                  alt="Kristijan Digital"
                  width={48}
                  height={48}
                  className="rounded-3xl shadow-2xl transition-transform duration-700 group-hover:scale-105 relative z-10 border border-white/10"
                />
                <div className="absolute inset-0 rounded-3xl bg-gradient-to-br from-white/10 to-transparent group-hover:from-white/20 transition-all duration-700"></div>
              </div>
              <div>
                <h1 className="text-2xl font-bold text-white">kristijan.digital</h1>
                <p className="text-xs text-white/60 font-medium">Digital Innovation Specialist</p>
              </div>
            </Link>

            {/* Desktop Navigation */}
            <nav className="hidden lg:flex items-center gap-12">
              {[
                { href: "/", label: "Home" },
                { href: "/#about", label: "About" },
                { href: "/#expertise", label: "Expertise" },
                { href: "/#projects", label: "Projects" },
                { href: "/#testimonials", label: "Testimonials" },
              ].map((item) => (
                <Link
                  key={item.label}
                  href={item.href}
                  className="text-sm font-medium text-white/70 hover:text-white hover:scale-110 transition-all duration-500 relative group"
                >
                  {item.label}
                  <div className="absolute -bottom-2 left-0 w-0 h-0.5 bg-gradient-to-r from-gray-400 to-gray-300 group-hover:w-full transition-all duration-500"></div>
                </Link>
              ))}
            </nav>

            <div className="flex items-center gap-6">
              <Button 
                className="hidden lg:flex bg-white/10 backdrop-blur-xl border-2 border-white/20 hover:bg-white/20 hover:border-white/40 shadow-2xl hover:shadow-white/20 transition-all duration-700 hover:scale-110 rounded-2xl px-8 py-3 font-semibold text-white"
                onClick={() => document.getElementById('contact-form')?.scrollIntoView({ behavior: 'smooth' })}
              >
                Get In Touch
                <Mail className="ml-2 h-4 w-4" />
              </Button>

              {/* Mobile Menu Button */}
              <Button
                variant="ghost"
                size="icon"
                className="lg:hidden rounded-2xl bg-white/5 backdrop-blur-xl border border-white/10 hover:bg-white/10 hover:border-white/30 transition-all duration-500"
                onClick={() => setIsMenuOpen(!isMenuOpen)}
              >
                {isMenuOpen ? <X className="h-5 w-5 text-white" /> : <Menu className="h-5 w-5 text-white" />}
              </Button>
            </div>
          </div>

          {/* Mobile Navigation */}
          {isMenuOpen && (
            <nav className="lg:hidden mt-6 pb-6 border-t border-white/10 pt-6 bg-black/20 backdrop-blur-3xl rounded-3xl border border-white/10 shadow-2xl">
              <div className="flex flex-col gap-6">
                {[
                  { href: "/", label: "Home" },
                  { href: "/#about", label: "About" },
                  { href: "/#expertise", label: "Expertise" },
                  { href: "/#projects", label: "Projects" },
                  { href: "/#testimonials", label: "Testimonials" },
                ].map((item) => (
                  <Link
                    key={item.label}
                    href={item.href}
                    className="text-sm font-medium text-white/80 hover:text-white transition-colors duration-300 px-4"
                  >
                    {item.label}
                  </Link>
                ))}
              </div>
            </nav>
          )}
        </div>
      </header>

      <section className="relative py-32 md:py-48 overflow-hidden bg-[#161d25]">
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_20%_80%,rgba(255,255,255,0.1),transparent_70%)]"></div>
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_80%_20%,rgba(255,255,255,0.08),transparent_70%)]"></div>
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_40%_40%,rgba(255,255,255,0.05),transparent_70%)]"></div>

        <div className="container mx-auto px-4 relative text-center z-10">
          <div className="mb-8">
            <div className="inline-flex items-center gap-3 bg-black/30 backdrop-blur-2xl border border-white/20 rounded-full px-6 py-3 mb-8 shadow-2xl">
              <div className="w-2 h-2 bg-green-400 rounded-full animate-pulse shadow-lg shadow-green-400/50"></div>
              <span className="text-sm font-medium text-white">Ready to start your project</span>
            </div>
          </div>

          <h1 className="text-5xl md:text-7xl lg:text-8xl font-bold leading-[0.9] text-balance mb-8 animate-fade-in">
            <span className="block text-white">Let's Create</span>
            <span className="block text-white">Something</span>
            <span className="block text-white">Amazing</span>
          </h1>

          <p className="text-xl md:text-2xl text-white/70 mb-12 text-pretty max-w-3xl mx-auto animate-slide-up leading-relaxed">
            Have a project in mind? I'd love to hear about it. Let's discuss how we can transform your digital vision
            into reality.
          </p>
        </div>
      </section>

      <section className="py-32 relative bg-white">
        <div className="container mx-auto px-4 relative z-10">
          <div className="grid lg:grid-cols-12 gap-16">
            {/* Contact Form */}
            <div className="lg:col-span-7">
              <Card className="animate-scale-in bg-white/90 backdrop-blur-sm border-2 border-gray-200/80 shadow-xl hover:shadow-2xl hover:shadow-gray-300/50 transition-all duration-700 rounded-3xl" id="contact-form">
                <CardHeader className="p-8">
                  <CardTitle className="text-3xl text-black font-bold mb-2">Start Your Project</CardTitle>
                  <p className="text-gray-600 text-lg">
                    Fill out the form below and I'll get back to you within 24 hours with a detailed proposal.
                  </p>
                </CardHeader>
                <CardContent className="p-8 pt-0">
                  <form onSubmit={handleSubmit} className="space-y-8">
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div className="space-y-3">
                        <Label htmlFor="firstName" className="text-gray-700 font-medium">
                          First Name *
                        </Label>
                        <Input
                          id="firstName"
                          name="firstName"
                          required
                          className="bg-gray-50 border-2 border-gray-200 focus:border-gray-400 focus:ring-2 focus:ring-gray-300/20 transition-all duration-300 rounded-2xl text-gray-900 placeholder:text-gray-400 h-12 hover:bg-gray-100"
                        />
                      </div>
                      <div className="space-y-3">
                        <Label htmlFor="lastName" className="text-gray-700 font-medium">
                          Last Name *
                        </Label>
                        <Input
                          id="lastName"
                          name="lastName"
                          required
                          className="bg-gray-50 border-2 border-gray-200 focus:border-gray-400 focus:ring-2 focus:ring-gray-300/20 transition-all duration-300 rounded-2xl text-gray-900 placeholder:text-gray-400 h-12 hover:bg-gray-100"
                        />
                      </div>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div className="space-y-3">
                        <Label htmlFor="email" className="text-gray-700 font-medium">
                          Email Address *
                        </Label>
                        <Input
                          id="email"
                          name="email"
                          type="email"
                          required
                          className="bg-gray-50 border-2 border-gray-200 focus:border-gray-400 focus:ring-2 focus:ring-gray-300/20 transition-all duration-300 rounded-2xl text-gray-900 placeholder:text-gray-400 h-12 hover:bg-gray-100"
                        />
                      </div>
                      <div className="space-y-3">
                        <Label htmlFor="phone" className="text-gray-700 font-medium">
                          Phone Number
                        </Label>
                        <Input
                          id="phone"
                          name="phone"
                          type="tel"
                          className="bg-gray-50 border-2 border-gray-200 focus:border-gray-400 focus:ring-2 focus:ring-gray-300/20 transition-all duration-300 rounded-2xl text-gray-900 placeholder:text-gray-400 h-12 hover:bg-gray-100"
                        />
                      </div>
                    </div>

                    <div className="space-y-3">
                      <Label htmlFor="service" className="text-gray-700 font-medium">
                        Service Interested In *
                      </Label>
                      <Select name="service" required>
                        <SelectTrigger className="bg-gray-50 border-2 border-gray-200 focus:border-gray-400 focus:ring-2 focus:ring-gray-300/20 transition-all duration-300 rounded-2xl text-gray-900 h-12 hover:bg-gray-100">
                          <SelectValue placeholder="Select a service" />
                        </SelectTrigger>
                        <SelectContent className="bg-white border-2 border-gray-200 rounded-2xl shadow-xl">
                          <SelectItem value="web-design">Web Design & Development</SelectItem>
                          <SelectItem value="ai-integration">AI Integration</SelectItem>
                          <SelectItem value="ui-ux-design">UI/UX Design</SelectItem>
                          <SelectItem value="e-commerce">E-commerce Solution</SelectItem>
                          <SelectItem value="digital-strategy">Digital Strategy</SelectItem>
                          <SelectItem value="performance-optimization">Performance Optimization</SelectItem>
                          <SelectItem value="consulting">Digital Consulting</SelectItem>
                          <SelectItem value="other">Other</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div className="space-y-3">
                        <Label htmlFor="budget" className="text-gray-700 font-medium">
                          Project Budget
                        </Label>
                        <Select name="budget">
                          <SelectTrigger className="bg-gray-50 border-2 border-gray-200 focus:border-gray-400 focus:ring-2 focus:ring-gray-300/20 transition-all duration-300 rounded-2xl text-gray-900 h-12 hover:bg-gray-100">
                            <SelectValue placeholder="Select budget range" />
                          </SelectTrigger>
                          <SelectContent className="bg-white border-2 border-gray-200 rounded-2xl shadow-xl">
                            <SelectItem value="under-2500">Under $2,500</SelectItem>
                            <SelectItem value="2500-5000">$2,500 - $5,000</SelectItem>
                            <SelectItem value="5000-10000">$5,000 - $10,000</SelectItem>
                            <SelectItem value="10000-25000">$10,000 - $25,000</SelectItem>
                            <SelectItem value="over-25000">Over $25,000</SelectItem>
                            <SelectItem value="custom">Let's Discuss</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                      <div className="space-y-3">
                        <Label htmlFor="timeline" className="text-gray-700 font-medium">
                          Project Timeline
                        </Label>
                        <Select name="timeline">
                          <SelectTrigger className="bg-gray-50 border-2 border-gray-200 focus:border-gray-400 focus:ring-2 focus:ring-gray-300/20 transition-all duration-300 rounded-2xl text-gray-900 h-12 hover:bg-gray-100">
                            <SelectValue placeholder="Select timeline" />
                          </SelectTrigger>
                          <SelectContent className="bg-white border-2 border-gray-200 rounded-2xl shadow-xl">
                            <SelectItem value="asap">ASAP</SelectItem>
                            <SelectItem value="1-month">Within 1 month</SelectItem>
                            <SelectItem value="2-3-months">2-3 months</SelectItem>
                            <SelectItem value="3-6-months">3-6 months</SelectItem>
                            <SelectItem value="6-plus-months">6+ months</SelectItem>
                            <SelectItem value="flexible">Flexible</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                    </div>

                    <div className="space-y-3">
                      <Label htmlFor="company" className="text-gray-700 font-medium">
                        Company Name
                      </Label>
                      <Input
                        id="company"
                        name="company"
                        placeholder="Optional"
                        className="bg-gray-50 border-2 border-gray-200 focus:border-gray-400 focus:ring-2 focus:ring-gray-300/20 transition-all duration-300 rounded-2xl text-gray-900 placeholder:text-gray-400 h-12 hover:bg-gray-100"
                      />
                    </div>

                    <div className="space-y-3">
                      <Label htmlFor="message" className="text-gray-700 font-medium">
                        Project Details *
                      </Label>
                      <Textarea
                        id="message"
                        name="message"
                        rows={6}
                        required
                        placeholder="Tell me about your project vision, goals, target audience, and any specific requirements or features you have in mind..."
                        className="bg-gray-50 border-2 border-gray-200 focus:border-gray-400 focus:ring-2 focus:ring-gray-300/20 transition-all duration-300 resize-none rounded-2xl text-gray-900 placeholder:text-gray-400 hover:bg-gray-100"
                      />
                    </div>

                    <div className="flex items-center space-x-3">
                      <Checkbox id="newsletter" name="newsletter" className="border-2 border-gray-300 rounded" />
                      <Label htmlFor="newsletter" className="text-sm text-gray-600">
                        I'd like to receive updates about your services and industry insights
                      </Label>
                    </div>

                    <Button
                      type="submit"
                      className="w-full bg-black hover:bg-gray-800 shadow-2xl hover:shadow-gray-500/30 transition-all duration-700 hover:scale-105 rounded-2xl h-14 text-lg font-semibold text-white"
                      disabled={isSubmitting}
                    >
                      {isSubmitting ? (
                        <>
                          <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white mr-3"></div>
                          Sending Your Message...
                        </>
                      ) : (
                        <>
                          Send Project Inquiry
                          <Send className="ml-3 h-5 w-5" />
                        </>
                      )}
                    </Button>

                    {showSuccess && (
                      <div className="flex items-center gap-3 text-green-700 bg-green-100 p-4 rounded-2xl border border-green-300 animate-fade-in">
                        <CheckCircle className="h-6 w-6" />
                        <span className="font-medium">
                          Message sent successfully! I'll get back to you within 24 hours with a detailed proposal.
                        </span>
                      </div>
                    )}
                  </form>
                </CardContent>
              </Card>
            </div>

            <div className="lg:col-span-5 space-y-8 animate-slide-up">
              <Card className="bg-white/90 backdrop-blur-sm border-2 border-gray-200/80 shadow-xl hover:shadow-2xl hover:shadow-gray-300/50 transition-all duration-700 hover:scale-105 rounded-3xl">
                <CardHeader className="p-8">
                  <CardTitle className="text-2xl text-black font-bold">Get In Touch</CardTitle>
                  <p className="text-gray-600">Prefer direct communication? Here are the best ways to reach me.</p>
                </CardHeader>
                <CardContent className="p-8 pt-0 space-y-8">
                  {[
                    {
                      icon: Mail,
                      title: "Email Address",
                      content: "contact@kristijan.digital",
                      description: "Best for detailed project discussions",
                      action: "Send Email",
                    },
                    {
                      icon: Phone,
                      title: "Phone & WhatsApp",
                      content: "+387 65 123 456",
                      description: "For urgent inquiries and consultations",
                      action: "Call Now",
                    },
                    {
                      icon: MapPin,
                      title: "Location",
                      content: "Banja Luka, Bosnia and Herzegovina",
                      description: "Available for local meetings",
                      action: null,
                    },
                    {
                      icon: Clock,
                      title: "Working Hours",
                      content: "Monday - Friday: 9:00 AM - 6:00 PM CET\nSaturday: By Appointment Only",
                      description: "Response within 24 hours guaranteed",
                      action: null,
                    },
                  ].map((item, index) => (
                    <div key={index} className="flex items-start gap-4 group">
                      <div className="bg-gray-100 p-4 rounded-3xl shadow-lg group-hover:shadow-gray-300/50 transition-all duration-500 group-hover:scale-125 border border-gray-200">
                        {React.createElement(item.icon, { className: "h-6 w-6 text-gray-700" })}
                      </div>
                      <div className="flex-1">
                        <h3 className="font-bold mb-2 text-black text-lg">{item.title}</h3>
                        <p className="text-gray-700 mb-1 whitespace-pre-line font-medium">{item.content}</p>
                        <p className="text-gray-500 text-sm mb-3">{item.description}</p>
                        {item.action && (
                          <Button
                            variant="ghost"
                            size="sm"
                            className="p-0 h-auto hover:text-gray-600 transition-colors duration-500 text-gray-700 font-medium"
                          >
                            {item.action}
                            <ArrowRight className="ml-2 h-3 w-3" />
                          </Button>
                        )}
                      </div>
                    </div>
                  ))}
                </CardContent>
              </Card>

              <Card className="bg-white/90 backdrop-blur-sm border-2 border-gray-200/80 shadow-xl hover:shadow-2xl hover:shadow-gray-300/50 transition-all duration-700 hover:scale-105 rounded-3xl">
                <CardHeader className="p-8">
                  <CardTitle className="text-black font-bold">Quick Connect</CardTitle>
                  <p className="text-gray-600">
                    For quick questions or to see my latest work, connect with me on social media.
                  </p>
                </CardHeader>
                <CardContent className="p-8 pt-0">
                  <div className="grid grid-cols-2 gap-4">
                    <Button
                      variant="outline"
                      className="bg-gray-50 border-2 border-gray-200 hover:bg-gray-100 hover:border-gray-300 hover:shadow-xl transition-all duration-500 hover:scale-110 rounded-2xl text-gray-700 h-14 font-medium"
                    >
                      <Instagram className="mr-2 h-5 w-5" />
                      Instagram
                    </Button>
                    <Button
                      variant="outline"
                      className="bg-gray-50 border-2 border-gray-200 hover:bg-gray-100 hover:border-gray-300 hover:shadow-xl transition-all duration-500 hover:scale-110 rounded-2xl text-gray-700 h-14 font-medium"
                    >
                      <Linkedin className="mr-2 h-5 w-5" />
                      LinkedIn
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        </div>
      </section>

      <section className="py-24 md:py-32 relative bg-white">
        <div className="container mx-auto px-4 relative z-10">
          <div className="text-center mb-16 md:mb-20">
            <div className="inline-flex items-center gap-2 bg-gray-100 px-4 py-2 rounded-full mb-6 border border-gray-200">
              <div className="w-2 h-2 bg-black rounded-full"></div>
              <span className="text-sm font-medium text-gray-700">FAQ</span>
            </div>
            <h2 className="text-4xl md:text-5xl lg:text-6xl font-bold mb-6 text-black leading-tight">
              Frequently Asked
              <br />
              <span className="text-gray-600">Questions</span>
            </h2>
            <div className="w-24 h-1 bg-gradient-to-r from-black via-gray-600 to-black mx-auto rounded-full shadow-lg mb-8"></div>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
              Get answers to common questions about my services, process, and approach to digital innovation
            </p>
          </div>

          <div className="max-w-4xl mx-auto space-y-3">
            {faqs.map((faq, index) => (
              <div key={index} className="group">
                <Card className="overflow-hidden bg-white border border-gray-200 shadow-md hover:shadow-lg hover:shadow-gray-300/30 transition-all duration-300 rounded-xl hover:scale-[1.01]">
                  <button
                    className="w-full p-4 md:p-5 text-left flex items-center justify-between hover:bg-gray-50/50 transition-all duration-200 group/button"
                    onClick={() => setOpenFaq(openFaq === index ? null : index)}
                  >
                    <span className="font-semibold text-black text-base md:text-lg pr-4 group-hover/button:text-gray-700 transition-colors duration-200">
                      {faq.question}
                    </span>
                    <div className="flex-shrink-0 relative">
                      <div className="w-8 h-8 bg-gray-100 rounded-full flex items-center justify-center group-hover/button:bg-gray-200 transition-colors duration-200">
                        <Plus
                          className={`h-4 w-4 absolute transition-all duration-300 ${
                            openFaq === index ? "rotate-45 opacity-0 scale-75" : "rotate-0 opacity-100 scale-100"
                          } text-gray-600`}
                        />
                        <Minus
                          className={`h-4 w-4 absolute transition-all duration-300 ${
                            openFaq === index ? "rotate-0 opacity-100 scale-100" : "rotate-45 opacity-0 scale-75"
                          } text-gray-600`}
                        />
                      </div>
                    </div>
                  </button>

                  <div
                    className="overflow-hidden transition-all duration-300 ease-in-out"
                    style={{ height: openFaq === index ? "auto" : "0px" }}
                  >
                    {openFaq === index && (
                      <div className="px-4 md:px-5 pb-4 md:pb-5">
                        <div className="bg-gray-50 p-4 md:p-5 rounded-lg border border-gray-100">
                          <p className="text-gray-700 leading-relaxed text-sm md:text-base">{faq.answer}</p>
                        </div>
                      </div>
                    )}
                  </div>
                </Card>
              </div>
            ))}
          </div>
        </div>
      </section>

      <footer className="py-16 bg-[#161d25]">
        <div className="container mx-auto px-4 text-center">
          <div className="flex justify-center gap-6 mb-8">
            {[
              { icon: Linkedin, label: "LinkedIn", href: "#" },
              { icon: Instagram, label: "Instagram", href: "#" },
              { icon: Mail, label: "Email", href: "mailto:contact@kristijan.digital" },
            ].map(({ icon: Icon, label, href }, index) => (
              <a key={label} href={href} className="group">
                <Button
                  variant="ghost"
                  size="icon"
                  className="rounded-3xl bg-white/5 backdrop-blur-3xl border border-white/10 hover:bg-white/10 hover:border-white/30 hover:shadow-2xl transition-all duration-500 hover:scale-125 w-16 h-16"
                  style={{ transitionDelay: `${index * 100}ms` }}
                >
                  <Icon className="h-7 w-7 text-white/80 group-hover:text-white transition-colors duration-500" />
                </Button>
              </a>
            ))}
          </div>

          <div className="text-sm text-white/50 bg-white/5 px-6 py-3 rounded-2xl backdrop-blur-sm inline-block border border-white/10">
            © 2024 Kristijan Design. All rights reserved.
          </div>
        </div>
      </footer>
    </div>
  )
}
