"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = __importDefault(require("path"));
const crypto_1 = __importDefault(require("crypto"));
const rollup_pluginutils_1 = require("rollup-pluginutils");
const sourcemap_codec_1 = require("sourcemap-codec");
function hash(content) {
    return crypto_1.default.createHmac('sha256', content)
        .digest('hex')
        .substr(0, 8);
}
function makeFileName(name, hashed, pattern) {
    return pattern.replace('[name]', name).replace('[hash]', hashed);
}
const cssChunks = function (options = {}) {
    const filter = rollup_pluginutils_1.createFilter(/\.css$/i, []);
    const defaultPluginOptions = {
        injectImports: false,
        ignore: false,
        sourcemap: false,
        chunkFileNames: '[name]-[hash].css',
        entryFileNames: '[name].css',
    };
    Object.keys(options).forEach(key => {
        if (!(key in defaultPluginOptions))
            throw new Error(`unknown option ${key}`);
    });
    const pluginOptions = Object.assign({}, defaultPluginOptions, options);
    const data = {
        css: {},
        map: {}
    };
    return {
        name: 'css',
        transform(code, id) {
            if (!filter(id))
                return null;
            if (pluginOptions.ignore !== false)
                return '';
            const m = code.match(/\/\*#\W*sourceMappingURL=data:application\/json;charset=utf-8;base64,([a-zA-Z0-9+/]+)\W*\*\//);
            if (m !== null) {
                data.css[id] = code.replace(m[0], '').trim();
                if (pluginOptions.sourcemap) {
                    try {
                        data.map[id] = JSON.parse(Buffer.from(m[1], 'base64').toString('utf-8').trim());
                    }
                    finally { // eslint-disable-line
                    }
                }
            }
            else {
                data.css[id] = code;
            }
            return { code: '', moduleSideEffects: 'no-treeshake' };
        },
        generateBundle(generateBundleOpts, bundle) {
            if (pluginOptions.ignore !== false)
                return;
            if (!generateBundleOpts.dir) {
                this.warn('No directory provided. Skipping CSS generation');
                return;
            }
            for (const chunk of Object.values(bundle).reverse()) {
                if (chunk.type === 'asset')
                    continue;
                let code = '';
                if (pluginOptions.injectImports) {
                    for (const c of chunk.imports) {
                        if (bundle[c]) {
                            code += bundle[c].imports.filter(filter)
                                .map(f => `@import '${f}';`).join('');
                        }
                    }
                }
                const sources = [];
                const sourcesContent = [];
                const mappings = [];
                for (const f of Object.keys(chunk.modules).filter(filter)) {
                    if (data.map[f]) {
                        const i = sources.length;
                        sources.push(path_1.default.relative(generateBundleOpts.dir, data.map[f].sources[0]));
                        sourcesContent.push(...data.map[f].sourcesContent);
                        const decoded = sourcemap_codec_1.decode(data.map[f].mappings);
                        if (i === 0) {
                            decoded[0].forEach(segment => {
                                segment[0] += code.length;
                            });
                        }
                        if (i > 0) {
                            decoded.forEach(line => {
                                line.forEach(segment => {
                                    segment[1] = i;
                                });
                            });
                        }
                        mappings.push(...decoded);
                    }
                    else if (pluginOptions.sourcemap) {
                        sources.push('');
                        sourcesContent.push('');
                        mappings.push(...(new Array(data.css[f].split(/\r\n|\r|\n/).length).fill([])));
                    }
                    code += data.css[f] + '\n';
                }
                if (code === '')
                    continue;
                const css_file_name = makeFileName(chunk.name, hash(code), chunk.isEntry ? pluginOptions.entryFileNames : pluginOptions.chunkFileNames);
                let map = null;
                if (mappings.length > 0) {
                    const map_file_name = css_file_name + '.map';
                    map = {
                        version: 3,
                        file: css_file_name,
                        sources: sources,
                        sourcesContent: sourcesContent,
                        names: [],
                        mappings: sourcemap_codec_1.encode(mappings)
                    };
                    code += `/*# sourceMappingURL=${encodeURIComponent(map_file_name)} */`;
                    this.emitFile({
                        type: 'asset',
                        fileName: map_file_name,
                        source: JSON.stringify(map, null)
                    });
                }
                this.emitFile({
                    type: 'asset',
                    fileName: css_file_name,
                    source: code
                });
                chunk.imports.push(css_file_name);
            }
        }
    };
};
exports.default = cssChunks;
