"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.HMR_OVERLAY_CODE = exports.HMR_CLIENT_CODE = exports.removeTrailingSlash = exports.removeLeadingSlash = exports.addTrailingSlash = exports.addLeadingSlash = exports.appendHtmlToHead = exports.relativeURL = exports.jsSourceMappingURL = exports.cssSourceMappingURL = exports.sanitizePackageName = exports.isRemoteUrl = exports.replaceExt = exports.getExt = exports.findMatchingAliasEntry = exports.clearCache = exports.updateLockfileHash = exports.checkLockfileHash = exports.openInBrowser = exports.MISSING_PLUGIN_SUGGESTIONS = exports.resolveDependencyManifest = exports.parsePackageImportSpecifier = exports.getPackageSource = exports.isTruthy = exports.writeLockfile = exports.readLockfile = exports.readFile = exports.SVELTE_VUE_REGEX = exports.CSS_REGEX = exports.HTML_JS_REGEX = exports.DEV_DEPENDENCIES_DIR = exports.PROJECT_CACHE_DIR = exports.BUILD_CACHE = exports.GLOBAL_CACHE_DIR = void 0;
const cacache_1 = __importDefault(require("cacache"));
const cachedir_1 = __importDefault(require("cachedir"));
const crypto_1 = __importDefault(require("crypto"));
const etag_1 = __importDefault(require("etag"));
const execa_1 = __importDefault(require("execa"));
const find_cache_dir_1 = __importDefault(require("find-cache-dir"));
const find_up_1 = __importDefault(require("find-up"));
const fs_1 = __importDefault(require("fs"));
const isbinaryfile_1 = require("isbinaryfile");
const mkdirp_1 = __importDefault(require("mkdirp"));
const open_1 = __importDefault(require("open"));
const path_1 = __importDefault(require("path"));
const rimraf_1 = __importDefault(require("rimraf"));
const skypack_1 = require("skypack");
const url_1 = __importDefault(require("url"));
const local_1 = __importDefault(require("./sources/local"));
const skypack_2 = __importDefault(require("./sources/skypack"));
exports.GLOBAL_CACHE_DIR = cachedir_1.default('snowpack');
// A note on cache naming/versioning: We currently version our global caches
// with the version of the last breaking change. This allows us to re-use the
// same cache across versions until something in the data structure changes.
// At that point, bump the version in the cache name to create a new unique
// cache name.
exports.BUILD_CACHE = path_1.default.join(exports.GLOBAL_CACHE_DIR, 'build-cache-2.7');
exports.PROJECT_CACHE_DIR = find_cache_dir_1.default({ name: 'snowpack' }) ||
    // If `projectCacheDir()` is null, no node_modules directory exists.
    // Use the current path (hashed) to create a cache entry in the global cache instead.
    // Because this is specifically for dependencies, this fallback should rarely be used.
    path_1.default.join(exports.GLOBAL_CACHE_DIR, crypto_1.default.createHash('md5').update(process.cwd()).digest('hex'));
exports.DEV_DEPENDENCIES_DIR = path_1.default.join(exports.PROJECT_CACHE_DIR, process.env.NODE_ENV || 'development');
const LOCKFILE_HASH_FILE = '.hash';
// NOTE(fks): Must match empty script elements to work properly.
exports.HTML_JS_REGEX = /(<script[^>]*?type="module".*?>)(.*?)<\/script>/gims;
exports.CSS_REGEX = /@import\s*['"](.*?)['"];/gs;
exports.SVELTE_VUE_REGEX = /(<script[^>]*>)(.*?)<\/script>/gims;
/** Read file from disk; return a string if it’s a code file */
async function readFile(filepath) {
    const data = await fs_1.default.promises.readFile(filepath);
    const isBinary = await isbinaryfile_1.isBinaryFile(data);
    return isBinary ? data : data.toString('utf-8');
}
exports.readFile = readFile;
async function readLockfile(cwd) {
    try {
        var lockfileContents = fs_1.default.readFileSync(path_1.default.join(cwd, 'snowpack.lock.json'), {
            encoding: 'utf-8',
        });
    }
    catch (err) {
        // no lockfile found, ignore and continue
        return null;
    }
    // If this fails, we actually do want to alert the user by throwing
    return JSON.parse(lockfileContents);
}
exports.readLockfile = readLockfile;
function sortObject(originalObject) {
    const newObject = {};
    for (const key of Object.keys(originalObject).sort()) {
        newObject[key] = originalObject[key];
    }
    return newObject;
}
async function writeLockfile(loc, importMap) {
    importMap.dependencies = sortObject(importMap.dependencies);
    importMap.imports = sortObject(importMap.imports);
    fs_1.default.writeFileSync(loc, JSON.stringify(importMap, undefined, 2), { encoding: 'utf-8' });
}
exports.writeLockfile = writeLockfile;
function isTruthy(item) {
    return Boolean(item);
}
exports.isTruthy = isTruthy;
function getPackageSource(source) {
    return source === 'local' ? local_1.default : skypack_2.default;
}
exports.getPackageSource = getPackageSource;
/** Get the package name + an entrypoint within that package (if given). */
function parsePackageImportSpecifier(imp) {
    const impParts = imp.split('/');
    if (imp.startsWith('@')) {
        const [scope, name, ...rest] = impParts;
        return [`${scope}/${name}`, rest.join('/') || null];
    }
    const [name, ...rest] = impParts;
    return [name, rest.join('/') || null];
}
exports.parsePackageImportSpecifier = parsePackageImportSpecifier;
/**
 * Given a package name, look for that package's package.json manifest.
 * Return both the manifest location (if believed to exist) and the
 * manifest itself (if found).
 *
 * NOTE: You used to be able to require() a package.json file directly,
 * but now with export map support in Node v13 that's no longer possible.
 */
function resolveDependencyManifest(dep, cwd) {
    // Attempt #1: Resolve the dependency manifest normally. This works for most
    // packages, but fails when the package defines an export map that doesn't
    // include a package.json. If we detect that to be the reason for failure,
    // move on to our custom implementation.
    try {
        const depManifest = fs_1.default.realpathSync.native(require.resolve(`${dep}/package.json`, { paths: [cwd] }));
        return [depManifest, require(depManifest)];
    }
    catch (err) {
        // if its an export map issue, move on to our manual resolver.
        if (err.code !== 'ERR_PACKAGE_PATH_NOT_EXPORTED') {
            return [null, null];
        }
    }
    // Attempt #2: Resolve the dependency manifest manually. This involves resolving
    // the dep itself to find the entrypoint file, and then haphazardly replacing the
    // file path within the package with a "./package.json" instead. It's not as
    // thorough as Attempt #1, but it should work well until export maps become more
    // established & move out of experimental mode.
    let result = [null, null];
    try {
        const fullPath = fs_1.default.realpathSync.native(require.resolve(dep, { paths: [cwd] }));
        // Strip everything after the package name to get the package root path
        // NOTE: This find-replace is very gross, replace with something like upath.
        const searchPath = `${path_1.default.sep}node_modules${path_1.default.sep}${dep.replace('/', path_1.default.sep)}`;
        const indexOfSearch = fullPath.lastIndexOf(searchPath);
        if (indexOfSearch >= 0) {
            const manifestPath = fullPath.substring(0, indexOfSearch + searchPath.length + 1) + 'package.json';
            result[0] = manifestPath;
            const manifestStr = fs_1.default.readFileSync(manifestPath, { encoding: 'utf-8' });
            result[1] = JSON.parse(manifestStr);
        }
    }
    catch (err) {
        // ignore
    }
    finally {
        return result;
    }
}
exports.resolveDependencyManifest = resolveDependencyManifest;
/**
 * If Rollup erred parsing a particular file, show suggestions based on its
 * file extension (note: lowercase is fine).
 */
exports.MISSING_PLUGIN_SUGGESTIONS = {
    '.svelte': 'Try installing rollup-plugin-svelte and adding it to Snowpack (https://www.snowpack.dev/#custom-rollup-plugins)',
    '.vue': 'Try installing rollup-plugin-vue and adding it to Snowpack (https://www.snowpack.dev/#custom-rollup-plugins)',
};
const appNames = {
    win32: {
        brave: 'brave',
        chrome: 'chrome',
    },
    darwin: {
        brave: 'Brave Browser',
        chrome: 'Google Chrome',
    },
    linux: {
        brave: 'brave',
        chrome: 'google-chrome',
    },
};
async function openInExistingChromeBrowser(url) {
    // see if Chrome process is open; fail if not
    await execa_1.default.command('ps cax | grep "Google Chrome"', {
        shell: true,
    });
    // use open Chrome tab if exists; create new Chrome tab if not
    const openChrome = execa_1.default('osascript ../assets/openChrome.appleScript "' + encodeURI(url) + '"', {
        cwd: __dirname,
        stdio: 'ignore',
        shell: true,
    });
    // if Chrome doesn’t respond within 3s, fall back to opening new tab in default browser
    let isChromeStalled = setTimeout(() => {
        openChrome.cancel();
    }, 3000);
    try {
        await openChrome;
    }
    catch (err) {
        if (err.isCanceled) {
            console.warn(`Chrome not responding to Snowpack after 3s. Opening in new tab.`);
        }
        else {
            console.error(err.toString() || err);
        }
        throw err;
    }
    finally {
        clearTimeout(isChromeStalled);
    }
}
async function openInBrowser(protocol, hostname, port, browser) {
    const url = `${protocol}//${hostname}:${port}`;
    browser = /chrome/i.test(browser)
        ? appNames[process.platform]['chrome']
        : /brave/i.test(browser)
            ? appNames[process.platform]['brave']
            : browser;
    const isMac = process.platform === 'darwin';
    const isBrowserChrome = /chrome|default/i.test(browser);
    if (!isMac || !isBrowserChrome) {
        await (browser === 'default' ? open_1.default(url) : open_1.default(url, { app: browser }));
        return;
    }
    try {
        // If we're on macOS, and we haven't requested a specific browser,
        // we can try opening Chrome with AppleScript. This lets us reuse an
        // existing tab when possible instead of creating a new one.
        await openInExistingChromeBrowser(url);
    }
    catch (err) {
        // if no open Chrome process, just go ahead and open default browser.
        await open_1.default(url);
    }
}
exports.openInBrowser = openInBrowser;
async function checkLockfileHash(dir) {
    const lockfileLoc = await find_up_1.default(['package-lock.json', 'yarn.lock']);
    if (!lockfileLoc) {
        return true;
    }
    const hashLoc = path_1.default.join(dir, LOCKFILE_HASH_FILE);
    const newLockHash = etag_1.default(await fs_1.default.promises.readFile(lockfileLoc, 'utf-8'));
    const oldLockHash = await fs_1.default.promises.readFile(hashLoc, 'utf-8').catch(() => '');
    return newLockHash === oldLockHash;
}
exports.checkLockfileHash = checkLockfileHash;
async function updateLockfileHash(dir) {
    const lockfileLoc = await find_up_1.default(['package-lock.json', 'yarn.lock']);
    if (!lockfileLoc) {
        return;
    }
    const hashLoc = path_1.default.join(dir, LOCKFILE_HASH_FILE);
    const newLockHash = etag_1.default(await fs_1.default.promises.readFile(lockfileLoc));
    await mkdirp_1.default(path_1.default.dirname(hashLoc));
    await fs_1.default.promises.writeFile(hashLoc, newLockHash);
}
exports.updateLockfileHash = updateLockfileHash;
async function clearCache() {
    return Promise.all([
        skypack_1.clearCache(),
        cacache_1.default.rm.all(exports.BUILD_CACHE),
        rimraf_1.default.sync(exports.PROJECT_CACHE_DIR),
    ]);
}
exports.clearCache = clearCache;
function getAliasType(val) {
    if (isRemoteUrl(val)) {
        return 'url';
    }
    return !path_1.default.isAbsolute(val) ? 'package' : 'path';
}
/**
 * For the given import specifier, return an alias entry if one is matched.
 */
function findMatchingAliasEntry(config, spec) {
    // Only match bare module specifiers. relative and absolute imports should not match
    if (spec === '.' ||
        spec === '..' ||
        spec.startsWith('./') ||
        spec.startsWith('../') ||
        spec.startsWith('/') ||
        spec.startsWith('http://') ||
        spec.startsWith('https://')) {
        return undefined;
    }
    for (const [from, to] of Object.entries(config.alias)) {
        const isExactMatch = spec === from;
        const isDeepMatch = spec.startsWith(addTrailingSlash(from));
        if (isExactMatch || isDeepMatch) {
            return {
                from,
                to,
                type: getAliasType(to),
            };
        }
    }
}
exports.findMatchingAliasEntry = findMatchingAliasEntry;
/** Get full extensions of files */
function getExt(fileName) {
    return {
        /** base extension (e.g. `.js`) */
        baseExt: path_1.default.extname(fileName).toLocaleLowerCase(),
        /** full extension, if applicable (e.g. `.proxy.js`) */
        expandedExt: path_1.default.basename(fileName).replace(/[^.]+/, '').toLocaleLowerCase(),
    };
}
exports.getExt = getExt;
/** Replace file extensions */
function replaceExt(fileName, oldExt, newExt) {
    const extToReplace = new RegExp(`\\${oldExt}$`, 'i');
    return fileName.replace(extToReplace, newExt);
}
exports.replaceExt = replaceExt;
function isRemoteUrl(val) {
    var _a;
    return val.startsWith('//') || !!((_a = url_1.default.parse(val).protocol) === null || _a === void 0 ? void 0 : _a.startsWith('http'));
}
exports.isRemoteUrl = isRemoteUrl;
/**
 * Sanitizes npm packages that end in .js (e.g `tippy.js` -> `tippyjs`).
 * This is necessary because Snowpack can’t create both a file and directory
 * that end in .js.
 */
function sanitizePackageName(filepath) {
    const dirs = filepath.split('/');
    const file = dirs.pop();
    return [...dirs.map((path) => path.replace(/\.js$/i, 'js')), file].join('/');
}
exports.sanitizePackageName = sanitizePackageName;
// Source Map spec v3: https://docs.google.com/document/d/1U1RGAehQwRypUTovF1KRlpiOFze0b-_2gc6fAH0KY0k/edit#heading=h.lmz475t4mvbx
/** CSS sourceMappingURL */
function cssSourceMappingURL(code, sourceMappingURL) {
    return code + `/*# sourceMappingURL=${sourceMappingURL} */`;
}
exports.cssSourceMappingURL = cssSourceMappingURL;
/** JS sourceMappingURL */
function jsSourceMappingURL(code, sourceMappingURL) {
    return code.replace(/\n*$/, '') + `\n//# sourceMappingURL=${sourceMappingURL}\n`; // strip ending lines & append source map (with linebreaks for safety)
}
exports.jsSourceMappingURL = jsSourceMappingURL;
/** URL relative */
function relativeURL(path1, path2) {
    let url = path_1.default.relative(path1, path2).replace(/\\/g, '/');
    if (!url.startsWith('./') && !url.startsWith('../')) {
        url = './' + url;
    }
    return url;
}
exports.relativeURL = relativeURL;
const CLOSING_HEAD_TAG = /<\s*\/\s*head\s*>/gi;
/** Append HTML before closing </head> tag */
function appendHtmlToHead(doc, htmlToAdd) {
    const closingHeadMatch = doc.match(CLOSING_HEAD_TAG);
    // if no <head> tag found, throw an error (we can’t load your app properly)
    if (!closingHeadMatch) {
        throw new Error(`No <head> tag found in HTML (this is needed to optimize your app):\n${doc}`);
    }
    // if multiple <head> tags found, also freak out
    if (closingHeadMatch.length > 1) {
        throw new Error(`Multiple <head> tags found in HTML (perhaps commented out?):\n${doc}`);
    }
    return doc.replace(closingHeadMatch[0], htmlToAdd + closingHeadMatch[0]);
}
exports.appendHtmlToHead = appendHtmlToHead;
/** Add / to beginning of string (but don’t double-up) */
function addLeadingSlash(path) {
    return path.replace(/^\/?/, '/');
}
exports.addLeadingSlash = addLeadingSlash;
/** Add / to the end of string (but don’t double-up) */
function addTrailingSlash(path) {
    return path.replace(/\/?$/, '/');
}
exports.addTrailingSlash = addTrailingSlash;
/** Remove \ and / from beginning of string */
function removeLeadingSlash(path) {
    return path.replace(/^[/\\]+/, '');
}
exports.removeLeadingSlash = removeLeadingSlash;
/** Remove \ and / from end of string */
function removeTrailingSlash(path) {
    return path.replace(/[/\\]+$/, '');
}
exports.removeTrailingSlash = removeTrailingSlash;
exports.HMR_CLIENT_CODE = fs_1.default.readFileSync(path_1.default.join(__dirname, '../assets/hmr-client.js'), 'utf-8');
exports.HMR_OVERLAY_CODE = fs_1.default.readFileSync(path_1.default.join(__dirname, '../assets/hmr-error-overlay.js'), 'utf-8');
//# sourceMappingURL=util.js.map