'use strict';

var _package = require('./package.js');
var path = require('path');
require('module');
var fs = require('fs');
var Url = require('url');
var child_process = require('child_process');
var utils = require('./utils.js');
var util = require('util');
var rollup = require('rollup');
var rollupPluginTerser = require('rollup-plugin-terser');
var css_chunks = require('rollup-plugin-css-chunks');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var path__default = /*#__PURE__*/_interopDefaultLegacy(path);
var fs__default = /*#__PURE__*/_interopDefaultLegacy(fs);
var child_process__default = /*#__PURE__*/_interopDefaultLegacy(child_process);
var css_chunks__default = /*#__PURE__*/_interopDefaultLegacy(css_chunks);

const inject_styles = `
export default function(files) {
	return Promise.all(files.map(function(file) { return new Promise(function(fulfil, reject) {
		var href = new URL(file, import.meta.url);
		var baseURI = document.baseURI;
		if (!baseURI) {
			var baseTags = document.getElementsByTagName('base');
			baseURI = baseTags.length ? baseTags[0].href : document.URL;
		}
		var relative = ('' + href).substring(baseURI.length);
		var link = document.querySelector('link[rel=stylesheet][href="' + relative + '"]')
			|| document.querySelector('link[rel=stylesheet][href="' + href + '"]');
		if (!link) {
			link = document.createElement('link');
			link.rel = 'stylesheet';
			link.href = href;
			document.head.appendChild(link);
		}
		if (link.sheet) {
			fulfil();
		} else {
			link.onload = function() { return fulfil() };
			link.onerror = reject;
		}
	})}));
};`.trim();

const INJECT_STYLES_NAME = 'inject_styles';
const INJECT_STYLES_ID = 'inject_styles.js';

const find_css = (chunk, bundle) => {
	const css_files = new Set();
	const visited = new Set();

	const recurse = (c) => {
		if (visited.has(c)) return;
		visited.add(c);

		if (c.imports) {
			c.imports.forEach((file) => {
				if (file.endsWith('.css')) {
					css_files.add(file);
				} else {
					const imported_chunk = bundle[file];
					if (imported_chunk) {
						recurse(imported_chunk);
					}
				}
			});
		}
	};

	recurse(chunk);
	return Array.from(css_files);
};

const css_injection = {
	name: 'svelte-css-injection',
	buildStart() {
		this.emitFile({
			type: 'chunk',
			id: INJECT_STYLES_ID,
			name: INJECT_STYLES_NAME,
			preserveSignature: 'allow-extension'
		});
	},
	load(id) {
		return id === INJECT_STYLES_ID ? inject_styles : null;
	},
	resolveId(importee) {
		return importee === INJECT_STYLES_ID ? INJECT_STYLES_ID : null;
	},
	renderDynamicImport({ targetModuleId }) {
		if (targetModuleId) {
			const t = Buffer.from(targetModuleId).toString('hex');
			return {
				left: 'Promise.all([import(',
				right: `), ___SVELTE_CSS_INJECTION___${t}___]).then(function(x) { return x[0]; })`
			};
		} else {
			return {
				left: 'import(',
				right: ')'
			};
		}
	},
	async generateBundle(
		
		_options,
		bundle
	) {
		const inject_styles_file = Object.keys(bundle).find((f) => f.startsWith('inject_styles'));

		let has_css = false;
		for (const name in bundle) {
			const chunk = bundle[name];

			let chunk_has_css = false;

			if (chunk.code) {
				chunk.code = chunk.code.replace(/___SVELTE_CSS_INJECTION___([0-9a-f]+)___/g, (_m, id) => {
					id = Buffer.from(id, 'hex').toString();
					const target = (
						Object.values(bundle).find(
							(c) => (c).modules && (c).modules[id]
						)
					);

					if (target) {
						const css_files = find_css(target, bundle);
						if (css_files.length > 0) {
							chunk_has_css = true;
							return `__inject_styles(${JSON.stringify(css_files)})`;
						}
					}

					return '';
				});

				if (chunk_has_css) {
					has_css = true;
					chunk.code += `\nimport __inject_styles from './${inject_styles_file}';`;
				}
			}
		}

		if (!has_css) {
			delete bundle[inject_styles_file];
		}

		// console.log(bundle);
	}
};

function clean_html(html) {
	return html
		.replace(/<!\[CDATA\[[\s\S]*?\]\]>/gm, '')
		.replace(/(<script[\s\S]*?>)[\s\S]*?<\/script>/gm, '$1</' + 'script>')
		.replace(/(<style[\s\S]*?>)[\s\S]*?<\/style>/gm, '$1</' + 'style>')
		.replace(/<!--[\s\S]*?-->/gm, '');
}

function get_href(attrs) {
	const match = /href\s*=\s*(?:"(.*?)"|'(.*?)'|([^\s>]*))/.exec(attrs);
	return match && (match[1] || match[2] || match[3]);
}

function get_src(attrs) {
	const match = /src\s*=\s*(?:"(.*?)"|'(.*?)'|([^\s>]*))/.exec(attrs);
	return match && (match[1] || match[2] || match[3]);
}

function get_srcset_urls(attrs) {
	const results = [];
	// Note that the srcset allows any ASCII whitespace, including newlines.
	const match = /srcset\s*=\s*(?:"(.*?)"|'(.*?)'|([^\s>]*))/s.exec(attrs);
	if (match) {
		const attr_content = match[1] || match[2] || match[3];
		// Parse the content of the srcset attribute.
		// The regexp is modelled after the srcset specs (https://html.spec.whatwg.org/multipage/images.html#srcset-attribute)
		// and should cover most reasonable cases.
		const regex = /\s*([^\s,]\S+[^\s,])\s*((?:\d+w)|(?:-?\d+(?:\.\d+)?(?:[eE]-?\d+)?x))?/gm;
		let sub_matches;
		while ((sub_matches = regex.exec(attr_content))) {
			results.push(sub_matches[1]);
		}
	}
	return results;
}

const OK = 2;
const REDIRECT = 3;

async function prerender({
	dir,
	out,
	manifest,
	log,
	force
}) {
	const seen = new Set();

	const server_root = path.resolve(dir);
	const app = require(`${server_root}/server/app.js`);

	async function crawl(path$1) {
		if (seen.has(path$1)) return;
		seen.add(path$1);

		const rendered = await app.render({
			host: null, // TODO ???
			method: 'GET',
			headers: {},
			path: path$1,
			body: null,
			query: new Url.URLSearchParams()
		}, {
			only_prerender: !force
		});

		if (rendered) {
			const response_type = Math.floor(rendered.status / 100);
			const headers = rendered.headers;
			const type = headers && headers['content-type'];
			const is_html = response_type === REDIRECT || type === 'text/html';

			const parts = path$1.split('/');
			if (is_html && parts[parts.length - 1] !== 'index.html') {
				parts.push('index.html');
			}

			const file = `${out}${parts.join('/')}`;
			utils.mkdirp(path.dirname(file));

			if (response_type === REDIRECT) {
				const location = headers['location'];

				log.warn(`${rendered.status} ${path$1} -> ${location}`);
				fs__default['default'].writeFileSync(
					file,
					`<script>window.location.href=${JSON.stringify(headers['location'])}</script>`
				);

				return;
			}

			if (response_type === OK) {
				log.info(`${rendered.status} ${path$1}`);
				fs__default['default'].writeFileSync(file, rendered.body); // TODO minify where possible?
			} else {
				// TODO should this fail the build?
				log.error(`${rendered.status} ${path$1}`);
			}

			const { dependencies } = rendered ;

			if (dependencies) {
				for (const path$1 in dependencies) {
					const result = dependencies[path$1];
					const response_type = Math.floor(result.status / 100);

					const is_html = result.headers['content-type'] === 'text/html';

					const parts = path$1.split('/');
					if (is_html && parts[parts.length - 1] !== 'index.html') {
						parts.push('index.html');
					}

					const file = `${out}${parts.join('/')}`;
					utils.mkdirp(path.dirname(file));

					fs__default['default'].writeFileSync(file, result.body);

					if (response_type === OK) {
						log.info(`${result.status} ${path$1}`);
					} else {
						log.error(`${result.status} ${path$1}`);
					}
				}
			}

			if (is_html) {
				const cleaned = clean_html(rendered.body);

				let match;
				const pattern = /<(a|img|link|source)\s+([\s\S]+?)>/gm;

				while ((match = pattern.exec(cleaned))) {
					let hrefs = [];
					const element = match[1];
					const attrs = match[2];

					if (element === 'a' || element === 'link') {
						hrefs.push(get_href(attrs));
					} else {
						if (element === 'img') {
							hrefs.push(get_src(attrs));
						}
						hrefs.push(...get_srcset_urls(attrs));
					}

					hrefs = hrefs.filter(Boolean);

					for (const href of hrefs) {
						const resolved = Url.resolve(path$1, href);
						if (resolved[0] !== '/') continue;

						const parsed = Url.parse(resolved);

						const parts = parsed.pathname.slice(1).split('/').filter(Boolean);
						if (parts[parts.length - 1] === 'index.html') parts.pop();

						const file_exists =
							(parsed.pathname.startsWith('/_app/') && fs__default['default'].existsSync(`${dir}/client/${parsed.pathname}`)) ||
							fs__default['default'].existsSync(`${out}${parsed.pathname}`) ||
							fs__default['default'].existsSync(`static${parsed.pathname}`) ||
							fs__default['default'].existsSync(`static${parsed.pathname}/index.html`);

						if (file_exists) continue;

						if (parsed.query) ;

						await crawl(parsed.pathname);
					}
				}
			}
		}
	}

	const entries = manifest.pages.map((page) => page.path).filter(Boolean);

	for (const entry of entries) {
		await crawl(entry);
	}
}

class Builder {
	#generated_files;
	#static_files;
	#manifest;

	constructor({
		generated_files,
		static_files,
		log,
		manifest
	}) {
		this.#generated_files = generated_files;
		this.#static_files = static_files;
		this.#manifest = manifest;

		this.log = log;
	}

	copy_client_files(dest) {
		utils.copy(`${this.#generated_files}/client`, dest, (file) => file[0] !== '.');
	}

	copy_server_files(dest) {
		utils.copy(`${this.#generated_files}/server`, dest, (file) => file[0] !== '.');
	}

	copy_static_files(dest) {
		utils.copy(this.#static_files, dest);
	}

	async prerender({
		force = false,
		dest
	}) {
		await prerender({
			out: dest,
			force,
			dir: this.#generated_files,
			manifest: this.#manifest,
			log: this.log
		});
	}
}

const exec = util.promisify(child_process__default['default'].exec);

const snowpack_main = require.resolve('snowpack');
const snowpack_pkg_file = path__default['default'].join(snowpack_main, '../../package.json');
const snowpack_pkg = require(snowpack_pkg_file); // eslint-disable-line
const snowpack_bin = path__default['default'].resolve(path__default['default'].dirname(snowpack_pkg_file), snowpack_pkg.bin.snowpack);

const ignorable_warnings = new Set(['EMPTY_BUNDLE', 'CIRCULAR_DEPENDENCY', 'MISSING_EXPORT']);
const onwarn = (warning, handler) => {
	// TODO would be nice to just eliminate the circular dependencies instead of
	// squelching these warnings (it happens when e.g. the root layout imports
	// from $app/navigation)
	if (ignorable_warnings.has(warning.code)) return;
	handler(warning);
};

async function build(config) {
	if (!config.adapter) {
		throw new Error('No adapter specified');
	}

	const manifest = utils.create_manifest_data(config.paths.routes);

	utils.mkdirp('.svelte/assets');
	utils.create_app({
		manifest_data: manifest,
		output: '.svelte/assets'
	});

	const header = (msg) => console.log(_package.$.bold().cyan(`\n> ${msg}`));

	const log = (msg) => console.log(msg.replace(/^/gm, '  '));
	log.success = (msg) => log(_package.$.green(`✔ ${msg}`));
	log.error = (msg) => log(_package.$.bold().red(msg));
	log.warn = (msg) => log(_package.$.bold().yellow(msg));
	log.minor = (msg) => log(_package.$.grey(msg));
	log.info = log;

	const unoptimized = '.svelte/build/unoptimized';

	{
		// phase one — build with Snowpack
		header('Creating unoptimized build...');
		await rimraf('.svelte/build/unoptimized');

		utils.copy_assets();

		const setup_file = `${unoptimized}/server/_app/setup/index.js`;
		if (!fs__default['default'].existsSync(setup_file)) {
			utils.mkdirp(path__default['default'].dirname(setup_file));
			fs__default['default'].writeFileSync(setup_file, '');
		}

		const mount = `--mount.${config.paths.routes}=/_app/routes --mount.${config.paths.setup}=/_app/setup`;

		await exec(`node ${snowpack_bin} build ${mount} --out=${unoptimized}/server --ssr`);
		log.success('server');
		await exec(`node ${snowpack_bin} build ${mount} --out=${unoptimized}/client`);
		log.success('client');
	}

	{
		// phase two — optimise
		header('Optimizing...');
		await rimraf('.svelte/build/optimized');

		const s = JSON.stringify;

		const client = {
			entry: null,
			deps: {}
		};

		const entry = path__default['default'].resolve(`${unoptimized}/client/_app/assets/runtime/internal/start.js`);

		// https://github.com/snowpackjs/snowpack/discussions/1395
		const re = /(\.\.\/)+_app\/assets\/app\//;
		const work_around_alias_bug = (type) => ({
			name: 'work-around-alias-bug',
			resolveId(imported) {
				if (re.test(imported)) {
					return path__default['default'].resolve(`${unoptimized}/${type}/_app/assets/app`, imported.replace(re, ''));
				}
			}
		});

		const client_chunks = await rollup.rollup({
			input: {
				entry
			},
			plugins: [
				work_around_alias_bug('client'),
				{
					name: 'deproxy-css',
					async resolveId(importee, importer) {
						if (/\.css\.proxy\.js$/.test(importee)) {
							const deproxied = importee.replace(/\.css\.proxy\.js$/, '.css');
							const resolved = await this.resolve(deproxied, importer);
							return resolved.id;
						}
					}
				},
				css_chunks__default['default']({
					injectImports: true,
					sourcemap: true
				}),
				css_injection,
				{
					name: 'generate-client-manifest',
					generateBundle(_options, bundle) {
						const reverse_lookup = new Map();

						const routes = path__default['default'].resolve(`${unoptimized}/client/_app/routes`);

						let inject_styles;

						for (const key in bundle) {
							const chunk = bundle[key];

							if (chunk.facadeModuleId === entry) {
								client.entry = key;
							} else if (chunk.facadeModuleId === 'inject_styles.js') {
								inject_styles = key;
							} else if (chunk.modules) {
								for (const id in chunk.modules) {
									if (id.startsWith(routes) && id.endsWith('.js')) {
										const file = id.slice(routes.length + 1);
										reverse_lookup.set(file, key);
									}
								}
							}
						}

						const find_deps = (key, js, css) => {
							if (js.has(key)) return;

							js.add(key);

							const chunk = bundle[key];

							if (chunk) {
								const imports = chunk.imports;

								if (imports) {
									imports.forEach((key) => {
										if (key.endsWith('.css')) {
											js.add(inject_styles);
											css.add(key);
										} else {
											find_deps(key, js, css);
										}
									});
								}
							} else {
								this.error(`'${key}' is imported but could not be bundled`);
							}

							return { js, css };
						};

						const get_deps = (key) => {
							const js = new Set();
							const css = new Set();

							find_deps(key, js, css);

							return {
								js: Array.from(js),
								css: Array.from(css)
							};
						};

						client.deps.__entry__ = get_deps(client.entry);

						manifest.components.forEach((component) => {
							const file = path__default['default'].normalize(component.file.replace(/\.svelte$/, '.js'));
							const key = reverse_lookup.get(file);

							client.deps[component.name] = get_deps(key);
						});
					}
				},
				rollupPluginTerser.terser()
			],

			onwarn,

			// TODO ensure this works with external node modules (on server)
			external: (id) => id[0] !== '.' && !path__default['default'].isAbsolute(id)
		});

		await client_chunks.write({
			dir: '.svelte/build/optimized/client/_app',
			entryFileNames: '[name]-[hash].js',
			chunkFileNames: '[name]-[hash].js',
			assetFileNames: '[name]-[hash].js', // TODO CSS filenames aren't hashed?
			format: 'esm',
			sourcemap: true
		});

		log.success('client');

		fs__default['default'].writeFileSync(`${unoptimized}/server/app.js`, `
			import * as renderer from '@sveltejs/kit/assets/renderer';
			import root from './_app/assets/generated/root.js';
			import * as setup from './_app/setup/index.js';

			const template = ({ head, body }) => ${s(fs__default['default'].readFileSync(config.paths.template, 'utf-8'))
				.replace('%svelte.head%', '" + head + "')
				.replace('%svelte.body%', '" + body + "')};

			const manifest = {
				layout: ${s(manifest.layout)},
				error: ${s(manifest.error)},
				components: ${s(manifest.components)},
				pages: [
					${manifest.pages
						.map(({ pattern, parts: json_parts }) => {
							const parts = JSON.stringify(json_parts);
							return `{ pattern: ${pattern}, parts: ${parts} }`;
						})
						.join(',')}
				],
				endpoints: [
					${manifest.endpoints
						.map(({ name, pattern, file, params: json_params }) => {
							const params = JSON.stringify(json_params);
							return `{ name: '${name}', pattern: ${pattern}, file: '${file}', params: ${params} }`;
						})
						.join(',')}
				]
			};

			const client = ${s(client)};

			export const paths = {
				static: ${s(config.paths.static)}
			};

			export function render(request, { only_prerender = false } = {}) {
				return renderer.render(request, {
					static_dir: paths.static,
					template,
					manifest,
					target: ${s(config.target)},
					client,
					root,
					setup,
					load: (route) => require(\`./routes/\${route.name}.js\`),
					dev: false,
					only_prerender
				});
			}
		`.replace(/^\t{3}/gm, '').trim());

		const server_input = {
			app: `${unoptimized}/server/app.js`
		};

		[
			manifest.layout,
			manifest.error,
			...manifest.components,
			...manifest.endpoints
		].forEach((item) => {
			server_input[`routes/${item.name}`] = `${unoptimized}/server${item.url.replace(
				/\.\w+$/,
				'.js'
			)}`;
		});

		const server_chunks = await rollup.rollup({
			input: server_input,
			plugins: [
				work_around_alias_bug('server'),
				{
					name: 'remove-css',
					load(id) {
						if (/\.css\.proxy\.js$/.test(id)) return '';
					}
				},
				// TODO add server manifest generation so we can prune
				// imports before zipping for cloud functions
				rollupPluginTerser.terser()
			],

			onwarn,

			// TODO ensure this works with external node modules (on server)
			external: (id) => id[0] !== '.' && !path__default['default'].isAbsolute(id)
		});

		await server_chunks.write({
			dir: '.svelte/build/optimized/server',
			format: 'cjs', // TODO some adapters might want ESM?
			exports: 'named',
			entryFileNames: '[name].js',
			chunkFileNames: 'chunks/[name].js',
			assetFileNames: 'assets/[name].js',
			sourcemap: true
		});

		log.success('server');
	}

	{
		// phase three — adapter
		header(`Generating app (${config.adapter})...`);

		const builder = new Builder({
			generated_files: '.svelte/build/optimized',
			static_files: config.paths.static,
			manifest,
			log
		});

		const adapter = _package.requireRelative_1(config.adapter);
		await adapter(builder);
	}

	log.success('done');
}

async function rimraf(path) {
	return new Promise((resolve) => {
		(fs__default['default'].rm || fs__default['default'].rmdir)(path, { recursive: true, force: true }, () => resolve());
	});
}

exports.build = build;
//# sourceMappingURL=index2.js.map
