"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.rollupPluginDependencyStats = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const zlib_1 = __importDefault(require("zlib"));
function rollupPluginDependencyStats(cb) {
    let outputDir;
    let existingFileCache = {};
    let statsSummary = {
        direct: {},
        common: {},
    };
    function buildExistingFileCache(bundle) {
        for (let fileName of Object.keys(bundle)) {
            const filePath = path_1.default.join(outputDir, fileName);
            if (fs_1.default.existsSync(filePath)) {
                const { size } = fs_1.default.statSync(filePath);
                existingFileCache[fileName] = size;
            }
        }
    }
    function compareDependencies(files, type) {
        for (let { fileName, contents } of files) {
            const size = contents.byteLength;
            statsSummary[type][fileName] = {
                size: size,
                gzip: zlib_1.default.gzipSync(contents).byteLength,
                brotli: zlib_1.default.brotliCompressSync ? zlib_1.default.brotliCompressSync(contents).byteLength : 0,
            };
            if (existingFileCache[fileName]) {
                const delta = (size - existingFileCache[fileName]) / 1000;
                statsSummary[type][fileName].delta = delta;
            }
        }
    }
    return {
        name: 'snowpack:rollup-plugin-stats',
        generateBundle(options, bundle) {
            outputDir = options.dir;
            buildExistingFileCache(bundle);
        },
        writeBundle(_, bundle) {
            const directDependencies = [];
            const commonDependencies = [];
            for (const [fileName, assetOrChunk] of Object.entries(bundle)) {
                const raw = assetOrChunk.type === 'asset' ? assetOrChunk.source : assetOrChunk.code;
                const contents = Buffer.isBuffer(raw)
                    ? raw
                    : typeof raw === 'string'
                        ? Buffer.from(raw, 'utf-8')
                        : Buffer.from(raw);
                if (fileName.startsWith('common')) {
                    commonDependencies.push({ fileName, contents });
                }
                else {
                    directDependencies.push({ fileName, contents });
                }
            }
            compareDependencies(directDependencies, 'direct');
            compareDependencies(commonDependencies, 'common');
            cb(statsSummary);
        },
    };
}
exports.rollupPluginDependencyStats = rollupPluginDependencyStats;
//# sourceMappingURL=rollup-plugin-stats.js.map