"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getWebDependencyType = exports.isJavaScript = exports.createInstallTarget = exports.removeTrailingSlash = exports.removeLeadingSlash = exports.addTrailingSlash = exports.addLeadingSlash = exports.getWebDependencyName = exports.sanitizePackageName = exports.getExt = exports.isPackageAliasEntry = exports.findMatchingAliasEntry = exports.MISSING_PLUGIN_SUGGESTIONS = exports.resolveDependencyManifest = exports.parsePackageImportSpecifier = exports.isTruthy = exports.isRemoteUrl = exports.writeLockfile = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const url_1 = __importDefault(require("url"));
const validate_npm_package_name_1 = __importDefault(require("validate-npm-package-name"));
async function writeLockfile(loc, importMap) {
    const sortedImportMap = { imports: {} };
    for (const key of Object.keys(importMap.imports).sort()) {
        sortedImportMap.imports[key] = importMap.imports[key];
    }
    fs_1.default.writeFileSync(loc, JSON.stringify(sortedImportMap, undefined, 2), { encoding: 'utf-8' });
}
exports.writeLockfile = writeLockfile;
function isRemoteUrl(val) {
    var _a;
    return val.startsWith('//') || !!((_a = url_1.default.parse(val).protocol) === null || _a === void 0 ? void 0 : _a.startsWith('http'));
}
exports.isRemoteUrl = isRemoteUrl;
function isTruthy(item) {
    return Boolean(item);
}
exports.isTruthy = isTruthy;
/** Get the package name + an entrypoint within that package (if given). */
function parsePackageImportSpecifier(imp) {
    const impParts = imp.split('/');
    if (imp.startsWith('@')) {
        const [scope, name, ...rest] = impParts;
        return [`${scope}/${name}`, rest.join('/') || null];
    }
    const [name, ...rest] = impParts;
    return [name, rest.join('/') || null];
}
exports.parsePackageImportSpecifier = parsePackageImportSpecifier;
/**
 * Given a package name, look for that package's package.json manifest.
 * Return both the manifest location (if believed to exist) and the
 * manifest itself (if found).
 *
 * NOTE: You used to be able to require() a package.json file directly,
 * but now with export map support in Node v13 that's no longer possible.
 */
function resolveDependencyManifest(dep, cwd) {
    // Attempt #1: Resolve the dependency manifest normally. This works for most
    // packages, but fails when the package defines an export map that doesn't
    // include a package.json. If we detect that to be the reason for failure,
    // move on to our custom implementation.
    try {
        const depManifest = fs_1.default.realpathSync.native(require.resolve(`${dep}/package.json`, { paths: [cwd] }));
        return [depManifest, require(depManifest)];
    }
    catch (err) {
        // if its an export map issue, move on to our manual resolver.
        if (err.code !== 'ERR_PACKAGE_PATH_NOT_EXPORTED') {
            return [null, null];
        }
    }
    // Attempt #2: Resolve the dependency manifest manually. This involves resolving
    // the dep itself to find the entrypoint file, and then haphazardly replacing the
    // file path within the package with a "./package.json" instead. It's not as
    // thorough as Attempt #1, but it should work well until export maps become more
    // established & move out of experimental mode.
    const result = [null, null];
    const fullPath = fs_1.default.realpathSync.native(require.resolve(dep, { paths: [cwd] }));
    // Strip everything after the package name to get the package root path
    // NOTE: This find-replace is very gross, replace with something like upath.
    const searchPath = `${path_1.default.sep}node_modules${path_1.default.sep}${dep.replace('/', path_1.default.sep)}`;
    const indexOfSearch = fullPath.lastIndexOf(searchPath);
    if (indexOfSearch >= 0) {
        const manifestPath = fullPath.substring(0, indexOfSearch + searchPath.length + 1) + 'package.json';
        result[0] = manifestPath;
        const manifestStr = fs_1.default.readFileSync(manifestPath, { encoding: 'utf-8' });
        result[1] = JSON.parse(manifestStr);
    }
    return result;
}
exports.resolveDependencyManifest = resolveDependencyManifest;
/**
 * If Rollup erred parsing a particular file, show suggestions based on its
 * file extension (note: lowercase is fine).
 */
exports.MISSING_PLUGIN_SUGGESTIONS = {
    '.svelte': 'Try installing rollup-plugin-svelte and adding it to Snowpack (https://www.snowpack.dev/#custom-rollup-plugins)',
    '.vue': 'Try installing rollup-plugin-vue and adding it to Snowpack (https://www.snowpack.dev/#custom-rollup-plugins)',
};
/**
 * For the given import specifier, return an alias entry if one is matched.
 */
function findMatchingAliasEntry(alias, spec) {
    // Only match bare module specifiers. relative and absolute imports should not match
    if (spec === '.' ||
        spec === '..' ||
        spec.startsWith('./') ||
        spec.startsWith('../') ||
        spec.startsWith('/') ||
        spec.startsWith('http://') ||
        spec.startsWith('https://')) {
        return undefined;
    }
    for (const [from, to] of Object.entries(alias)) {
        let foundType = isPackageAliasEntry(to) ? 'package' : 'path';
        const isExactMatch = spec === removeTrailingSlash(from);
        const isDeepMatch = spec.startsWith(addTrailingSlash(from));
        if (isExactMatch || isDeepMatch) {
            return {
                from,
                to,
                type: foundType,
            };
        }
    }
}
exports.findMatchingAliasEntry = findMatchingAliasEntry;
/**
 * For the given import specifier, return an alias entry if one is matched.
 */
function isPackageAliasEntry(val) {
    return !path_1.default.isAbsolute(val);
}
exports.isPackageAliasEntry = isPackageAliasEntry;
/** Get full extensions of files */
function getExt(fileName) {
    return {
        /** base extension (e.g. `.js`) */
        baseExt: path_1.default.extname(fileName).toLocaleLowerCase(),
        /** full extension, if applicable (e.g. `.proxy.js`) */
        expandedExt: path_1.default.basename(fileName).replace(/[^.]+/, '').toLocaleLowerCase(),
    };
}
exports.getExt = getExt;
/**
 * Sanitizes npm packages that end in .js (e.g `tippy.js` -> `tippyjs`).
 * This is necessary because Snowpack can’t create both a file and directory
 * that end in .js.
 */
function sanitizePackageName(filepath) {
    const dirs = filepath.split('/');
    const file = dirs.pop();
    return [...dirs.map((path) => path.replace(/\.js$/i, 'js')), file].join('/');
}
exports.sanitizePackageName = sanitizePackageName;
/**
 * Formats the snowpack dependency name from a "webDependencies" input value:
 * 2. Remove any ".js"/".mjs" extension (will be added automatically by Rollup)
 */
function getWebDependencyName(dep) {
    return validate_npm_package_name_1.default(dep).validForNewPackages
        ? dep.replace(/\.js$/i, 'js') // if this is a top-level package ending in .js, replace with js (e.g. tippy.js -> tippyjs)
        : dep.replace(/\.m?js$/i, ''); // otherwise simply strip the extension (Rollup will resolve it)
}
exports.getWebDependencyName = getWebDependencyName;
/** Add / to beginning of string (but don’t double-up) */
function addLeadingSlash(path) {
    return path.replace(/^\/?/, '/');
}
exports.addLeadingSlash = addLeadingSlash;
/** Add / to the end of string (but don’t double-up) */
function addTrailingSlash(path) {
    return path.replace(/\/?$/, '/');
}
exports.addTrailingSlash = addTrailingSlash;
/** Remove \ and / from beginning of string */
function removeLeadingSlash(path) {
    return path.replace(/^[/\\]+/, '');
}
exports.removeLeadingSlash = removeLeadingSlash;
/** Remove \ and / from end of string */
function removeTrailingSlash(path) {
    return path.replace(/[/\\]+$/, '');
}
exports.removeTrailingSlash = removeTrailingSlash;
function createInstallTarget(specifier, all = true) {
    return {
        specifier,
        all,
        default: false,
        namespace: false,
        named: [],
    };
}
exports.createInstallTarget = createInstallTarget;
function isJavaScript(pathname) {
    const ext = path_1.default.extname(pathname).toLowerCase();
    return ext === '.js' || ext === '.mjs' || ext === '.cjs';
}
exports.isJavaScript = isJavaScript;
/**
 * Detect the web dependency "type" as either JS or ASSET:
 *   - BUNDLE: Install and bundle this file with Rollup engine.
 *   - ASSET: Copy this file directly.
 */
function getWebDependencyType(pathname) {
    const ext = path_1.default.extname(pathname).toLowerCase();
    // JavaScript should always be bundled.
    if (isJavaScript(pathname)) {
        return 'BUNDLE';
    }
    // Svelte & Vue should always be bundled because we want to show the missing plugin
    // error if a Svelte or Vue file is the install target. Without this, the .svelte/.vue
    // file would be treated like an asset and sent to the web as-is.
    if (ext === '.svelte' || ext === '.vue') {
        return 'BUNDLE';
    }
    // All other files should be treated as assets (copied over directly).
    return 'ASSET';
}
exports.getWebDependencyType = getWebDependencyType;
//# sourceMappingURL=util.js.map