"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.clearCache = exports.lookupBySpecifier = exports.buildNewPackage = exports.fetchCDN = exports.generateImportMap = exports.SKYPACK_ORIGIN = exports.rollupPluginSkypack = void 0;
const cacache_1 = __importDefault(require("cacache"));
const got_1 = __importDefault(require("got"));
const util_1 = require("./util");
var rollup_plugin_remote_cdn_1 = require("./rollup-plugin-remote-cdn");
Object.defineProperty(exports, "rollupPluginSkypack", { enumerable: true, get: function () { return rollup_plugin_remote_cdn_1.rollupPluginSkypack; } });
var util_2 = require("./util");
Object.defineProperty(exports, "SKYPACK_ORIGIN", { enumerable: true, get: function () { return util_2.SKYPACK_ORIGIN; } });
function parseRawPackageImport(spec) {
    const impParts = spec.split('/');
    if (spec.startsWith('@')) {
        const [scope, name, ...rest] = impParts;
        return [`${scope}/${name}`, rest.join('/') || null];
    }
    const [name, ...rest] = impParts;
    return [name, rest.join('/') || null];
}
async function generateImportMap(webDependencies, inheritFromImportMap) {
    const newLockfile = inheritFromImportMap ? { imports: { ...inheritFromImportMap.imports } } : { imports: {} };
    await Promise.all(Object.entries(webDependencies).map(async ([packageName, packageSemver]) => {
        if (packageSemver === null) {
            delete newLockfile.imports[packageName];
            delete newLockfile.imports[packageName + '/'];
            return;
        }
        const lookupResponse = await lookupBySpecifier(packageName, packageSemver);
        if (lookupResponse.error) {
            throw lookupResponse.error;
        }
        if (lookupResponse.pinnedUrl) {
            let keepGoing = true;
            const deepPinnedUrlParts = lookupResponse.pinnedUrl.split('/');
            // TODO: Get ?meta support to get this info via JSON instead of header manipulation
            deepPinnedUrlParts.shift(); // remove ""
            deepPinnedUrlParts.shift(); // remove "pin"
            while (keepGoing) {
                const investigate = deepPinnedUrlParts.pop();
                if (util_1.HAS_CDN_HASH_REGEX.test(investigate)) {
                    keepGoing = false;
                    deepPinnedUrlParts.push(investigate);
                }
            }
            newLockfile.imports[packageName] = util_1.SKYPACK_ORIGIN + '/' + deepPinnedUrlParts.join('/');
            newLockfile.imports[packageName + '/'] = util_1.SKYPACK_ORIGIN + '/' + deepPinnedUrlParts.join('/') + '/';
        }
    }));
    const newLockfileSorted = Object.keys(newLockfile.imports).sort((a, b) => {
        // We want 'xxx/' to come after 'xxx', so we convert it to a space (the character with the highest sort order)
        // See: http://support.ecisolutions.com/doc-ddms/help/reportsmenu/ascii_sort_order_chart.htm
        return a.replace(/\/$/, ' ').localeCompare(b.replace(/\/$/, ' '));
    });
    return {
        imports: newLockfileSorted.reduce((prev, k) => {
            prev[k] = newLockfile.imports[k];
            return prev;
        }, {}),
    };
}
exports.generateImportMap = generateImportMap;
async function fetchCDN(resourceUrl, userAgent) {
    var _a;
    if (!resourceUrl.startsWith(util_1.SKYPACK_ORIGIN)) {
        resourceUrl = util_1.SKYPACK_ORIGIN + resourceUrl;
    }
    const cachedResult = await cacache_1.default.get(util_1.RESOURCE_CACHE, resourceUrl).catch(() => null);
    if (cachedResult) {
        const cachedResultMetadata = cachedResult.metadata;
        const freshUntil = new Date(cachedResult.metadata.freshUntil);
        if (freshUntil >= new Date()) {
            return {
                isCached: true,
                isStale: false,
                body: cachedResult.data.toString(),
                headers: cachedResultMetadata.headers,
                statusCode: cachedResultMetadata.statusCode,
            };
        }
    }
    let freshResult;
    try {
        freshResult = await got_1.default(resourceUrl, {
            headers: { 'user-agent': userAgent || `skypack/v0.0.1` },
            throwHttpErrors: false,
        });
    }
    catch (err) {
        if (cachedResult) {
            const cachedResultMetadata = cachedResult.metadata;
            return {
                isCached: true,
                isStale: true,
                body: cachedResult.data.toString(),
                headers: cachedResultMetadata.headers,
                statusCode: cachedResultMetadata.statusCode,
            };
        }
        throw err;
    }
    const cacheUntilMatch = (_a = freshResult.headers['cache-control']) === null || _a === void 0 ? void 0 : _a.match(/max-age=(\d+)/);
    if (cacheUntilMatch) {
        var freshUntil = new Date();
        freshUntil.setSeconds(freshUntil.getSeconds() + parseInt(cacheUntilMatch[1]));
        // no need to await, since we `.catch()` to swallow any errors.
        cacache_1.default
            .put(util_1.RESOURCE_CACHE, resourceUrl, freshResult.body, {
            metadata: {
                headers: freshResult.headers,
                statusCode: freshResult.statusCode,
                freshUntil: freshUntil.toUTCString(),
            },
        })
            .catch(() => null);
    }
    return {
        body: freshResult.body,
        headers: freshResult.headers,
        statusCode: freshResult.statusCode,
        isCached: false,
        isStale: false,
    };
}
exports.fetchCDN = fetchCDN;
async function buildNewPackage(spec, semverString, userAgent) {
    const [packageName, packagePath] = parseRawPackageImport(spec);
    const lookupUrl = `/new/${packageName}` +
        (semverString ? `@${semverString}` : ``) +
        (packagePath ? `/${packagePath}` : ``);
    try {
        const { statusCode } = await fetchCDN(lookupUrl, userAgent);
        return {
            error: null,
            success: statusCode !== 500,
        };
    }
    catch (err) {
        return { error: err, success: false };
    }
}
exports.buildNewPackage = buildNewPackage;
async function lookupBySpecifier(spec, semverString, userAgent) {
    const [packageName, packagePath] = parseRawPackageImport(spec);
    const lookupUrl = `/${packageName}` +
        (semverString ? `@${semverString}` : ``) +
        (packagePath ? `/${packagePath}` : ``);
    try {
        const { body, statusCode, headers, isCached, isStale } = await fetchCDN(lookupUrl, userAgent);
        if (statusCode !== 200) {
            return { error: new Error(body) };
        }
        return {
            error: null,
            body,
            isCached,
            isStale,
            importStatus: headers['x-import-status'],
            importUrl: headers['x-import-url'],
            pinnedUrl: headers['x-pinned-url'],
            typesUrl: headers['x-typescript-types'],
        };
    }
    catch (err) {
        return { error: err };
    }
}
exports.lookupBySpecifier = lookupBySpecifier;
async function clearCache() {
    return Promise.all([cacache_1.default.rm.all(util_1.RESOURCE_CACHE)]);
}
exports.clearCache = clearCache;
//# sourceMappingURL=index.js.map