"use client"

import React from "react"
import { Button } from "@/components/ui/button"
import { Card } from "@/components/ui/card"
import Link from "next/link"
import { Menu, X, ArrowRight, Code, Palette, Zap, Brain, Rocket, Target, CheckCircle } from "lucide-react"
import Image from "next/image"

export default function ServicesPage() {
  const [isMenuOpen, setIsMenuOpen] = React.useState(false)

  const services = [
    {
      slug: "web-design",
      icon: Palette,
      title: "Web Design & UI/UX",
      shortDesc: "User-centric design that converts",
      description:
        "Creating stunning, user-centric websites that captivate audiences and drive conversions. Every design is meticulously crafted to reflect your brand identity while ensuring optimal user experience across all devices.",
      features: [
        "Responsive Design",
        "User Experience Optimization",
        "Brand Identity Integration",
        "Conversion-Focused Layouts",
      ],
      pricing: "Starting at $2,500",
    },
    {
      slug: "web-development",
      icon: Code,
      title: "Web Development",
      shortDesc: "Modern, scalable solutions",
      description:
        "Building robust, scalable web applications using cutting-edge technologies and best practices. From simple websites to complex platforms, ensuring fast, secure, and maintainable digital solutions.",
      features: ["Modern Frameworks", "Performance Optimization", "Security Implementation", "Scalable Architecture"],
      pricing: "Starting at $3,000",
    },
    {
      slug: "ai-integration",
      icon: Brain,
      title: "AI Integration",
      shortDesc: "Intelligent automation solutions",
      description:
        "Harnessing artificial intelligence to create smart, adaptive digital solutions. Implementing AI-powered features like chatbots, predictive analytics, and automation that add real business value.",
      features: ["Chatbot Development", "Predictive Analytics", "Process Automation", "Machine Learning Models"],
      pricing: "Starting at $4,000",
    },
    {
      slug: "ecommerce",
      icon: Zap,
      title: "E-commerce Solutions",
      shortDesc: "Powerful online stores",
      description:
        "Developing powerful online stores with advanced features, secure payment systems, and optimized user journeys that maximize conversions and provide seamless shopping experiences.",
      features: ["Payment Integration", "Inventory Management", "Order Processing", "Analytics Dashboard"],
      pricing: "Starting at $3,500",
    },
    {
      slug: "digital-strategy",
      icon: Target,
      title: "Digital Strategy",
      shortDesc: "Data-driven growth approach",
      description:
        "Providing strategic guidance to help businesses navigate the digital landscape effectively. Data-driven approach ensures digital investments deliver measurable results and sustainable growth.",
      features: ["Market Analysis", "Competitive Research", "Growth Planning", "ROI Optimization"],
      pricing: "Starting at $2,000",
    },
    {
      slug: "performance-optimization",
      icon: Rocket,
      title: "Performance Optimization",
      shortDesc: "Maximum speed and efficiency",
      description:
        "Optimizing websites and applications for maximum speed, efficiency, and user satisfaction. Advanced techniques and monitoring ensure your digital presence performs at its peak potential.",
      features: ["Speed Optimization", "SEO Enhancement", "Core Web Vitals", "Performance Monitoring"],
      pricing: "Starting at $1,500",
    },
  ]

  return (
    <div className="min-h-screen bg-white text-gray-900">
      <header className="sticky top-0 z-50 backdrop-blur-3xl bg-[#161d25]/95 border-b border-white/10 shadow-2xl">
        <div className="container mx-auto px-4 py-6 relative">
          <div className="flex items-center justify-between">
            <Link href="/" className="flex items-center gap-4">
              <div className="relative group">
                <Image
                  src="https://i.ibb.co/5XLC8VTC/kdsgnpng2.png"
                  alt="Kristijan Digital"
                  width={40}
                  height={40}
                  className="rounded-2xl shadow-lg transition-transform duration-700 group-hover:scale-105 relative z-10"
                />
              </div>
              <div>
                <h1 className="text-2xl font-bold text-white">kristijan.digital</h1>
                <p className="text-xs text-white/60 font-medium">Digital Innovation Specialist</p>
              </div>
            </Link>

            {/* Desktop Navigation */}
            <nav className="hidden lg:flex items-center gap-12">
              {[
                { href: "/", label: "Home" },
                { href: "/#about", label: "About" },
                { href: "/services", label: "Services" },
                { href: "/#projects", label: "Projects" },
                { href: "/contact", label: "Contact" },
              ].map((item) => (
                <Link
                  key={item.label}
                  href={item.href}
                  className="text-sm font-medium text-white/70 hover:text-white hover:scale-110 transition-all duration-500 relative group"
                >
                  {item.label}
                  <div className="absolute -bottom-2 left-0 w-0 h-0.5 bg-gradient-to-r from-gray-400 to-gray-300 group-hover:w-full transition-all duration-500"></div>
                </Link>
              ))}
            </nav>

            <div className="flex items-center gap-6">
              <Link href="/contact">
                <Button className="hidden lg:flex bg-gradient-to-r from-white/15 to-white/10 backdrop-blur-xl border border-white/30 hover:bg-gradient-to-r hover:from-white/25 hover:to-white/20 hover:border-white/50 shadow-xl hover:shadow-2xl hover:shadow-white/20 transition-all duration-500 hover:scale-105 rounded-xl px-8 py-3 font-semibold text-white">
                  Get Started
                  <ArrowRight className="ml-2 h-4 w-4" />
                </Button>
              </Link>

              {/* Mobile Menu Button */}
              <Button
                variant="ghost"
                size="icon"
                className="lg:hidden rounded-2xl bg-white/5 backdrop-blur-xl border border-white/10 hover:bg-white/10 hover:border-white/30 transition-all duration-500"
                onClick={() => setIsMenuOpen(!isMenuOpen)}
              >
                {isMenuOpen ? <X className="h-5 w-5 text-white" /> : <Menu className="h-5 w-5 text-white" />}
              </Button>
            </div>
          </div>

          {/* Mobile Navigation */}
          {isMenuOpen && (
            <nav className="lg:hidden mt-6 pb-6 border-t border-white/10 pt-6 bg-black/20 backdrop-blur-3xl rounded-3xl border border-white/10 shadow-2xl">
              <div className="flex flex-col gap-6">
                {[
                  { href: "/", label: "Home" },
                  { href: "/#about", label: "About" },
                  { href: "/services", label: "Services" },
                  { href: "/#projects", label: "Projects" },
                  { href: "/contact", label: "Contact" },
                ].map((item) => (
                  <Link
                    key={item.label}
                    href={item.href}
                    className="text-sm font-medium text-white/80 hover:text-white transition-colors duration-300 px-4"
                  >
                    {item.label}
                  </Link>
                ))}
              </div>
            </nav>
          )}
        </div>
      </header>

      {/* Hero Section */}
      <section className="relative py-32 md:py-48 overflow-hidden bg-[#161d25]">
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_20%_80%,rgba(255,255,255,0.1),transparent_70%)]"></div>
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_80%_20%,rgba(255,255,255,0.08),transparent_70%)]"></div>

        <div className="container mx-auto px-4 relative z-10 text-center">
          <div className="mb-8">
            <div className="inline-flex items-center gap-3 bg-black/30 backdrop-blur-2xl border border-white/20 rounded-full px-6 py-3 mb-8 shadow-2xl">
              <div className="w-2 h-2 bg-blue-400 rounded-full animate-pulse shadow-lg shadow-blue-400/50"></div>
              <span className="text-sm font-medium text-white">Professional Services</span>
            </div>
          </div>

          <h1 className="text-5xl md:text-7xl lg:text-8xl font-bold leading-[0.9] text-balance mb-8">
            <span className="block text-white">Digital</span>
            <span className="block text-white">Services</span>
          </h1>

          <p className="text-xl md:text-2xl text-white/70 mb-12 text-pretty leading-relaxed max-w-3xl mx-auto">
            Comprehensive digital solutions designed to transform your business and drive measurable results through
            innovative technology and strategic design.
          </p>
        </div>
      </section>

      {/* Services Grid */}
      <section className="py-32 relative bg-white">
        <div className="container mx-auto px-4 relative z-10">
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {services.map((service, index) => (
              <Card
                key={service.slug}
                className="p-8 bg-white border-2 border-gray-200 shadow-xl hover:shadow-2xl hover:shadow-gray-300/50 transition-all duration-700 hover:scale-105 hover:-translate-y-4 group rounded-3xl relative overflow-hidden"
              >
                <div className="relative z-10">
                  <div className="bg-gray-100 p-4 rounded-3xl shadow-lg group-hover:shadow-gray-300/50 transition-all duration-700 group-hover:scale-125 group-hover:rotate-12 border border-gray-200 mb-6 w-fit">
                    {React.createElement(service.icon, { className: "h-8 w-8 text-gray-700" })}
                  </div>

                  <h3 className="text-2xl font-bold mb-4 group-hover:text-gray-600 transition-colors duration-500 text-black">
                    {service.title}
                  </h3>

                  <p className="text-gray-600 leading-relaxed mb-6">{service.description}</p>

                  <div className="mb-6">
                    <h4 className="font-semibold text-black mb-3">Key Features:</h4>
                    <ul className="space-y-2">
                      {service.features.map((feature, idx) => (
                        <li key={idx} className="flex items-center gap-2 text-sm text-gray-600">
                          <CheckCircle className="h-4 w-4 text-green-500" />
                          {feature}
                        </li>
                      ))}
                    </ul>
                  </div>

                  <div className="mb-6">
                    <div className="text-lg font-bold text-black">{service.pricing}</div>
                  </div>

                  <Link href={`/services/${service.slug}`}>
                    <Button className="w-full bg-black hover:bg-gray-800 shadow-lg hover:shadow-xl transition-all duration-500 hover:scale-105 rounded-2xl text-white">
                      Learn More
                      <ArrowRight className="ml-2 h-4 w-4" />
                    </Button>
                  </Link>
                </div>

                <div className="absolute inset-0 bg-gradient-to-br from-gray-50/50 via-gray-100/30 to-gray-50/50 opacity-0 group-hover:opacity-100 transition-all duration-700 rounded-3xl"></div>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-32 relative overflow-hidden bg-[#161d25]">
        <div className="container mx-auto px-4 relative z-10 text-center">
          <h2 className="text-4xl md:text-6xl font-bold mb-6 text-white leading-tight">Ready to Get Started?</h2>
          <p className="text-xl text-white/70 max-w-2xl mx-auto leading-relaxed mb-12">
            Let's discuss your project and find the perfect solution for your business needs.
          </p>
          <Link href="/contact">
            <Button
              size="lg"
              className="bg-gradient-to-r from-white/15 to-white/10 backdrop-blur-xl border border-white/30 hover:border-white/50 text-white rounded-xl px-12 py-4 text-lg font-semibold transition-all duration-500 hover:scale-105 hover:shadow-2xl hover:shadow-white/20"
            >
              Start Your Project
              <ArrowRight className="ml-3 h-5 w-5" />
            </Button>
          </Link>
        </div>
      </section>
    </div>
  )
}
